const SETTINGS_KEYS = ['apiKey', 'model', 'defaultEmail', 'defaultPassword', 'knowledgeBase', 'isActive'];

const els = {};

document.addEventListener('DOMContentLoaded', () => {
  els.apiKey = document.getElementById('apiKey');
  els.apiKeyToggle = document.getElementById('apiKeyToggle');
  els.eyeIconShow = document.getElementById('eyeIconShow');
  els.eyeIconHide = document.getElementById('eyeIconHide');
  els.modelSelect = document.getElementById('modelSelect');
  els.refreshModelsBtn = document.getElementById('refreshModelsBtn');
  els.defaultEmail = document.getElementById('defaultEmail');
  els.defaultPassword = document.getElementById('defaultPassword');
  els.knowledgeBase = document.getElementById('knowledgeBase');
  els.toggleActive = document.getElementById('toggleActive');
  els.toggleLabel = document.getElementById('toggleLabel');
  els.fillFormsBtn = document.getElementById('fillFormsBtn');
  els.saveSettingsBtn = document.getElementById('saveSettingsBtn');
  els.statusMessage = document.getElementById('statusMessage');
  els.wordCount = document.getElementById('wordCount');

  loadSettings();

  els.apiKeyToggle.addEventListener('click', toggleApiKeyVisibility);
  els.toggleActive.addEventListener('change', handleToggleChange);
  els.fillFormsBtn.addEventListener('click', handleFillForms);
  els.saveSettingsBtn.addEventListener('click', () => {
    saveSettings().then(() => setStatus('Settings saved.', 'success'));
  });
  els.knowledgeBase.addEventListener('input', updateWordCount);
  els.refreshModelsBtn.addEventListener('click', fetchModelsFromApi);
});

// Listen for status updates from background
chrome.runtime.onMessage.addListener((message) => {
  if (message.action === 'FILL_STATUS') {
    setStatus(message.text, message.type);
    if (message.type !== 'info') {
      els.fillFormsBtn.classList.remove('loading');
      els.fillFormsBtn.disabled = !els.toggleActive.checked;
    }
  }
});

function loadSettings() {
  chrome.storage.local.get(SETTINGS_KEYS, (result) => {
    if (result.apiKey) els.apiKey.value = result.apiKey;
    if (result.model) els.modelSelect.value = result.model;
    if (result.defaultEmail) els.defaultEmail.value = result.defaultEmail;
    if (result.defaultPassword) els.defaultPassword.value = result.defaultPassword;
    if (result.knowledgeBase) els.knowledgeBase.value = result.knowledgeBase;

    els.toggleActive.checked = !!result.isActive;
    updateToggleLabel();
    updateFillButtonState();
    updateWordCount();

    // Auto-fetch models if API key is present
    if (result.apiKey) fetchModelsFromApi();
  });
}

function saveSettings() {
  return new Promise((resolve) => {
    chrome.storage.local.set({
      apiKey: els.apiKey.value.trim(),
      model: els.modelSelect.value,
      defaultEmail: els.defaultEmail.value.trim(),
      defaultPassword: els.defaultPassword.value,
      knowledgeBase: els.knowledgeBase.value,
      isActive: els.toggleActive.checked
    }, resolve);
  });
}

function handleToggleChange() {
  updateToggleLabel();
  updateFillButtonState();
  saveSettings();
}

function updateToggleLabel() {
  els.toggleLabel.textContent = els.toggleActive.checked ? 'Active' : 'Inactive';
}

function updateFillButtonState() {
  els.fillFormsBtn.disabled = !els.toggleActive.checked;
}

async function handleFillForms() {
  if (!els.apiKey.value.trim()) {
    setStatus('Please enter your API key.', 'error');
    return;
  }
  if (!els.knowledgeBase.value.trim()) {
    setStatus('Please enter your knowledge base text.', 'error');
    return;
  }

  await saveSettings();

  els.fillFormsBtn.disabled = true;
  els.fillFormsBtn.classList.add('loading');
  setStatus('Analyzing form fields...', 'info');

  chrome.runtime.sendMessage({ action: 'FILL_FORMS' }, (response) => {
    els.fillFormsBtn.classList.remove('loading');
    els.fillFormsBtn.disabled = !els.toggleActive.checked;

    if (chrome.runtime.lastError) {
      setStatus('Extension error: ' + chrome.runtime.lastError.message, 'error');
      return;
    }

    if (response && response.success) {
      setStatus(`Filled ${response.filledCount} field(s) successfully.`, 'success');
    } else if (response) {
      setStatus(response.error || 'Unknown error occurred.', 'error');
    }
  });
}

function toggleApiKeyVisibility() {
  const isPassword = els.apiKey.type === 'password';
  els.apiKey.type = isPassword ? 'text' : 'password';
  els.eyeIconShow.classList.toggle('hidden', isPassword);
  els.eyeIconHide.classList.toggle('hidden', !isPassword);
}

function updateWordCount() {
  const text = els.knowledgeBase.value.trim();
  const count = text ? text.split(/\s+/).length : 0;
  els.wordCount.textContent = `${count.toLocaleString()} word${count !== 1 ? 's' : ''}`;
}

function setStatus(message, type) {
  els.statusMessage.textContent = message;
  els.statusMessage.className = `status ${type}`;
}

// ============================================================
// DYNAMIC MODEL FETCHING via ListModels API
// ============================================================

async function fetchModelsFromApi() {
  const apiKey = els.apiKey.value.trim();
  if (!apiKey) {
    setStatus('Enter your API key first to fetch models.', 'error');
    return;
  }

  els.refreshModelsBtn.disabled = true;
  els.refreshModelsBtn.textContent = 'Loading...';

  try {
    const resp = await fetch(
      `https://generativelanguage.googleapis.com/v1beta/models?key=${apiKey}&pageSize=100`
    );
    if (!resp.ok) {
      const err = await resp.json().catch(() => ({}));
      throw new Error(err.error?.message || `HTTP ${resp.status}`);
    }
    const data = await resp.json();

    // Filter to models that support generateContent
    const models = (data.models || []).filter(m =>
      m.supportedGenerationMethods?.includes('generateContent')
    );

    if (!models.length) {
      setStatus('No generateContent models found.', 'error');
      return;
    }

    // Sort: gemini-3 first, then 2.5, then 2.0, then rest
    models.sort((a, b) => {
      const pa = modelPriority(a.name);
      const pb = modelPriority(b.name);
      if (pa !== pb) return pa - pb;
      return a.displayName.localeCompare(b.displayName);
    });

    // Remember current selection
    const currentModel = els.modelSelect.value;

    // Rebuild select
    els.modelSelect.innerHTML = '';
    const groups = {};
    for (const m of models) {
      const id = m.name.replace('models/', '');
      const group = getModelGroup(id);
      if (!groups[group]) {
        groups[group] = document.createElement('optgroup');
        groups[group].label = group;
        els.modelSelect.appendChild(groups[group]);
      }
      const opt = document.createElement('option');
      opt.value = id;
      opt.textContent = m.displayName || id;
      groups[group].appendChild(opt);
    }

    // Restore selection or default to gemini-2.5-flash
    if (currentModel && els.modelSelect.querySelector(`option[value="${currentModel}"]`)) {
      els.modelSelect.value = currentModel;
    } else {
      const preferred = ['gemini-2.5-flash', 'gemini-2.0-flash'];
      for (const p of preferred) {
        if (els.modelSelect.querySelector(`option[value="${p}"]`)) {
          els.modelSelect.value = p;
          break;
        }
      }
    }

    setStatus(`Loaded ${models.length} models from API.`, 'success');
  } catch (err) {
    setStatus('Failed to fetch models: ' + err.message, 'error');
  } finally {
    els.refreshModelsBtn.disabled = false;
    els.refreshModelsBtn.textContent = 'Refresh';
  }
}

function modelPriority(name) {
  if (name.includes('gemini-3')) return 1;
  if (name.includes('gemini-2.5')) return 2;
  if (name.includes('gemini-2.0')) return 3;
  if (name.includes('gemini-1.5')) return 4;
  return 5;
}

function getModelGroup(id) {
  if (id.includes('gemini-3')) return 'Gemini 3';
  if (id.includes('gemini-2.5')) return 'Gemini 2.5';
  if (id.includes('gemini-2.0')) return 'Gemini 2.0';
  if (id.includes('gemini-1.5')) return 'Gemini 1.5';
  return 'Other';
}
